{******************************************************************************}
{                                                                              }
{                                MiniASMInterpreter                            }
{                                                                              }
{   Class interpreting a given MiniASM-source-code. Derived from               }
{   TMiniASMSourceWorker.                                                      }
{                                                                              }
{   original author:   Christian Rehn aka r2c2                                 }
{                                                                              }
{   This code is licenced under the terms of the Mozilla Public Licence(MPL)   }
{   See http://www.mozilla.org/MPL/ or licence-file included in surce-archive  }
{   for licence terms.                                                         }
{                                                                              }
{******************************************************************************}

//TODO: ggf. CaseSensitive-Wechsel erlauben

unit MiniAsmInterpreter;

interface

uses
  MiniAsmStack, MiniAsmSourceWorker, Classes, Syntax;

type
  TRegisterChangeEvent = procedure(Sender: TObject; AData: Integer) of object;
  TFlagChangeEvent = procedure(Sender: TObject; AData: Boolean) of object;
  TReadIntEvent = procedure(Sender: TObject; out AValue: Integer) of object;

  TMiniAsmInterpreter = class(TMiniAsmSourceWorker)
  strict private
    FStack: TMiniAsmStack;
    FTerminated: Boolean;
    // Register:
    Feax: Integer;
    Feip: Integer;
    // Flags:
    Fsf: Boolean;
    Fzf: Boolean;
    // events:
    FOnEAXChange: TRegisterChangeEvent;
    FOnZFChange: TFlagChangeEvent;
    FOnEIPChange: TRegisterChangeEvent;
    FOnSFChange: TFlagChangeEvent;
    FOnInterpretationComplete: TNotifyEvent;
    FOnReadInt: TReadIntEvent;
  protected
    procedure SetOnStackChange(const Value: TStackChangeEvent); virtual;
    function GetOnStackChange: TStackChangeEvent; virtual;
    procedure SetEAX(const Value: Integer); virtual;
    procedure SetEIP(const Value: Integer); virtual;
    procedure SetSF(const Value: Boolean); virtual;
    procedure SetZF(const Value: Boolean); virtual;
    procedure SetSource(ASource: TStrings); override;
    function DoRead(AOutput: TStrings; AArg: Integer): Boolean; virtual;
    function DoWrite(AOutput: TStrings; AArg: Integer): Boolean; virtual;
    function DoLoad(AOutput: TStrings; AArg: Integer): Boolean; virtual;
    function DoStore(AOutput: TStrings; AArg: Integer): Boolean; virtual;
    function DoAdd(AOutput: TStrings; AArg: Integer): Boolean; virtual;
    function DoSub(AOutput: TStrings; AArg: Integer): Boolean; virtual;
    function DoMul(AOutput: TStrings; AArg: Integer): Boolean; virtual;
    function DoDiv(AOutput: TStrings; AArg: Integer): Boolean; virtual;
    function DoJmp(AOutput: TStrings; AArg: Integer): Boolean; virtual;
    function DoJz(AOutput: TStrings; AArg: Integer): Boolean; virtual;
    function DoJns(AOutput: TStrings; AArg: Integer): Boolean; virtual;
    function DoPrint(AOutput: TStrings; AArg: string): Boolean; virtual;
    function DoData(AOutput: TStrings; AArg: Integer): Boolean; virtual;
    function DoPush(AOutput: TStrings): Boolean; virtual;
    function DoPop(AOutput: TStrings): Boolean; virtual;
    function DoGoSub(AOutput: TStrings; AArg: Integer): Boolean; virtual;
    function DoRet(AOutput: TStrings): Boolean; virtual;
  public
    constructor Create(ASource: TStrings; ASyntax: TSyntax);
    destructor Destroy; override;

    property Source write SetSource;
    property Stack: TMiniAsmStack read FStack;
    property Terminated: Boolean read FTerminated write FTerminated;
    // Register:
    property EAX: Integer read Feax write SetEAX; // Akkumulator
    property EIP: Integer read Feip write SetEIP;  // InstructionPointer; nchste Codezeile
    // Flags:
    property ZF: Boolean read Fzf write SetZF; // ZeroFlag
    property SF: Boolean read Fsf write SetSF; // SignFlag;
    // events:
    property OnStackChange: TStackChangeEvent read GetOnStackChange write SetOnStackChange;
    property OnEAXChange: TRegisterChangeEvent read FOnEAXChange write FOnEAXChange;
    property OnEIPChange: TRegisterChangeEvent read FOnEIPChange write FOnEIPChange;
    property OnZFChange: TFlagChangeEvent read FOnZFChange write FOnZFChange;
    property OnSFChange: TFlagChangeEvent read FOnSFChange write FOnSFChange;
    property OnInterpretationComplete: TNotifyEvent
      read FOnInterpretationComplete write FOnInterpretationComplete;
    property OnReadInt: TReadIntEvent read FOnReadInt write FOnReadInt;

    function Execute(AOutput: TStrings): Boolean; override;
    function Step(AOutput: TStrings): Boolean; virtual;
  end;

implementation

uses
  SysUtils, Forms;

{ TMiniAsmInterpreter }

constructor TMiniAsmInterpreter.Create(ASource: TStrings; ASyntax: TSyntax);
begin
  inherited Create(ASource, ASyntax);
  FStack := TMiniAsmStack.Create(ASyntax.StackSize);
  EIP := 0;
  PrepareSource;
  FTerminated := False;
end;

destructor TMiniAsmInterpreter.Destroy;
begin
  FStack.Free;
  inherited Destroy
end;

function TMiniAsmInterpreter.Execute(AOutput: TStrings): Boolean;
var
  cmd: TCommand;
  Operation: string;
  Arg: string;
begin
  Result := True;
  EIP := 0;
  if EIP < Source.Count then
  begin
    repeat
      ParseLine(EIP, Operation, Arg);

      cmd := Syntax.CreateCommand(Operation, Arg);
      if cmd <> nil then  // Leerzeilen berlesen
      begin
        case cmd.Operation of
          opRead: Result := DoRead(AOutput, cmd.Arg_i);
          opWrite: Result := DoWrite(AOutput, cmd.Arg_i);
          opLoad: Result := DoLoad(AOutput, cmd.Arg_i);
          opStore: Result := DoStore(AOutput, cmd.Arg_i);
          opAdd: Result := DoAdd(AOutput, cmd.Arg_i);
          opSub: Result := DoSub(AOutput, cmd.Arg_i);
          opMul: Result := DoMul(AOutput, cmd.Arg_i);
          opDiv: Result := DoDiv(AOutput, cmd.Arg_i);
          opJmp: Result := DoJmp(AOutput, cmd.Arg_i);
          opJz: Result := DoJz(AOutput, cmd.Arg_i);
          opJns: Result := DoJns(AOutput, cmd.Arg_i);
          opPrint: Result := DoPrint(AOutput, cmd.Arg_s);
          opData: Result := DoData(AOutput, cmd.Arg_i);
          opPush: Result := DoPush(AOutput);
          opPop: Result := DoPop(AOutput);
          opGoSub: Result := DoGoSub(AOutput, cmd.Arg_i);
          opRet: Result := DoRet(AOutput);
        end;
      end;
      EIP := EIP + 1;
      Application.ProcessMessages;   // for Termination
    until ((cmd <> nil) and (cmd.Operation = opEnd)) // Leerzeichen berlesen; Reihenfolge wichtig!
      or (not Result) or (EIP >= Source.Count) or FTerminated;
  end;

  FTerminated := False; // reset terminated, so the next interpretation is not
                        // terminated directly
  if Assigned(FOnInterpretationComplete) then
    FOnInterpretationComplete(Self);
end;

function TMiniAsmInterpreter.Step(AOutput: TStrings): Boolean;
var
  cmd: TCommand;
  Operation: string;
  Arg: string;
begin
  //Leerzeilen berlesen:
  while (EIP < Source.Count) and (Trim(Source[EIP]) = '') do
  begin
    EIP := EIP + 1;
  end;

  if (EIP < Source.Count) and (not FTerminated) then
  begin
    Result := True;
    ParseLine(EIP, Operation, Arg);

    cmd := Syntax.CreateCommand(Operation, Arg);
    if cmd <> nil then    // Leerzeilen berlesen
    begin
      case cmd.Operation of
        opRead: Result := DoRead(AOutput, cmd.Arg_i);
        opWrite: Result := DoWrite(AOutput, cmd.Arg_i);
        opLoad: Result := DoLoad(AOutput, cmd.Arg_i);
        opStore: Result := DoStore(AOutput, cmd.Arg_i);
        opAdd: Result := DoAdd(AOutput, cmd.Arg_i);
        opSub: Result := DoSub(AOutput, cmd.Arg_i);
        opMul: Result := DoMul(AOutput, cmd.Arg_i);
        opDiv: Result := DoDiv(AOutput, cmd.Arg_i);
        opJmp: Result := DoJmp(AOutput, cmd.Arg_i);
        opJz: Result := DoJz(AOutput, cmd.Arg_i);
        opJns: Result := DoJns(AOutput, cmd.Arg_i);
        opPrint: Result := DoPrint(AOutput, cmd.Arg_s);
        opData: Result := DoData(AOutput, cmd.Arg_i);
        opPush: Result := DoPush(AOutput);
        opPop: Result := DoPop(AOutput);
        opGoSub: Result := DoGoSub(AOutput, cmd.Arg_i);
        opRet: Result := DoRet(AOutput);
        opEnd:
        begin
          if Assigned(FOnInterpretationComplete) then
            FOnInterpretationComplete(Self);
          Exit;
        end;
      end;
    end;
    EIP := EIP + 1;
  end
  else
  begin
    Result := False;
    FTerminated := False; // reset terminated, so the next interpretation is not
                          // terminated directly
  end;
  if not Result then
  begin
    if Assigned(FOnInterpretationComplete) then
      FOnInterpretationComplete(Self);
  end;          
end;

{$REGION 'DoCommand'}

function TMiniAsmInterpreter.DoRead(AOutput: TStrings; AArg: Integer): Boolean;
var
  Value: Integer;
begin
  Result := True;
  try
    if Assigned(FOnReadInt) then
    begin
      FOnReadInt(Self, Value);
      Stack[AArg] := Value;
    end
    else
    begin
      raise Exception.Create('Lesen nicht mglich, da OnReadInt undefiniert ist.');
    end;
  except
    Result := False;
  end;
end;

function TMiniAsmInterpreter.DoWrite(AOutput: TStrings; AArg: Integer): Boolean;
begin
  Result := True;
  try
    AOutput.Add(Format('%d: %s %d' + #9 + '<Stack[%d]> = %d', [EIP,
      Syntax.WriteInt, AArg, AArg, FStack[AArg]]));
  except
    Result := False;
  end;
end;

function TMiniAsmInterpreter.DoLoad(AOutput: TStrings; AArg: Integer): Boolean;
begin
  Result := True;
  try
    EAX := FStack[AArg];
    AOutput.Add(Format('%d: %s %d' + #9 + '<AC> = %d', [EIP, Syntax.Load, AArg,
      EAX]));
  except
    Result := False;
  end;
end;

function TMiniAsmInterpreter.DoStore(AOutput: TStrings; AArg: Integer): Boolean;
begin
  Result := True;
  try
    FStack[AArg] := EAX;
    AOutput.Add(Format('%d: %s %d', [EIP, Syntax.Store, AArg]));
  except
    Result := False;
  end;
end;

function TMiniAsmInterpreter.DoAdd(AOutput: TStrings; AArg: Integer): Boolean;
begin
  Result := True;
  try
    EAX := EAX + FStack[AArg];
    AOutput.Add(Format('%d: %s %d' + #9 + '<AC> = %d', [EIP, Syntax.Add, AArg,
      EAX]));
    ZF := (EAX = 0);
    SF := (EAX < 0);
  except
    Result := False;
  end;
end;

function TMiniAsmInterpreter.DoSub(AOutput: TStrings; AArg: Integer): Boolean;
begin
  Result := True;
  try
    EAX := EAX - FStack[AArg];
    AOutput.Add(Format('%d: %s %d' + #9 + '<AC> = %d', [EIP, Syntax.Substract, AArg,
      EAX]));
    ZF := (EAX = 0);
    SF := (EAX < 0);
  except
    Result := False;
  end;
end;

function TMiniAsmInterpreter.DoMul(AOutput: TStrings; AArg: Integer): Boolean;
begin
  Result := True;
  try
    EAX := EAX * FStack[AArg];
    AOutput.Add(Format('%d: %s %d' + #9 + '<AC> = %d', [EIP, Syntax.Mutiply, AArg,
      EAX]));
    ZF := (EAX = 0);
    SF := (EAX < 0);
  except
    Result := False;
  end;
end;

function TMiniAsmInterpreter.DoDiv(AOutput: TStrings; AArg: Integer): Boolean;
begin
  Result := True;
  try
    EAX := EAX div FStack[AArg];
    AOutput.Add(Format('%d: %s %d' + #9 + '<AC> = %d', [EIP, Syntax.Devide, AArg,
      EAX]));
    ZF := (EAX = 0);
    SF := (EAX < 0);      
  except
    on EDivByZero do
    begin
      AOutput.Add('Fehler: Division durch 0. Programm angehalten.')
    end;
    on Exception do
    begin
      Result := False;
    end;
  end;
end;

function TMiniAsmInterpreter.DoJmp(AOutput: TStrings; AArg: Integer): Boolean;
begin
  Result := True;
  try
    AOutput.Add(Format('%d: %s %d', [EIP, Syntax.Jmp, AArg]));
    EIP := AArg - 1; // -1, da spter EIP wieder inkrementiert wird
  except
    Result := False;
  end;
end;

function TMiniAsmInterpreter.DoJz(AOutput: TStrings; AArg: Integer): Boolean;
begin
  Result := True;
  try
    if ZF then
    begin
      AOutput.Add(Format('%d: %s %d', [EIP, Syntax.Jz, AArg]));
      EIP := AArg - 1; // -1, da spter EIP wieder inkrementiert wird
    end;
  except
    Result := False;
  end;
end;

function TMiniAsmInterpreter.DoJns(AOutput: TStrings; AArg: Integer): Boolean;
begin
  Result := True;
  try
    if not SF then
    begin
      AOutput.Add(Format('%d: %s %d', [EIP, Syntax.Jns, AArg]));
      EIP := AArg - 1; // -1, da spter EIP wieder inkrementiert wird
    end;
  except
    Result := False;
  end;
end;

function TMiniAsmInterpreter.DoPrint(AOutput: TStrings; AArg: string): Boolean;
begin
  Result := True;
  try
    AOutput.Add(AArg);
  except
    Result := False;
  end;
end;

function TMiniAsmInterpreter.DoData(AOutput: TStrings; AArg: Integer): Boolean;
begin
  Result := True;
  try
    EAX := AArg;
    AOutput.Add(Format('%d: %s %d' + #9 + '<AC> = %d', [EIP, Syntax.Data, AArg,
      EAX]));
  except
    Result := False;
  end;
end;

function TMiniAsmInterpreter.DoPop(AOutput: TStrings): Boolean;
begin
  Result := True;
  try
    EAX := FStack.Pop;
    AOutput.Add(Format('%d: %s' + #9 + '<AC> = %d', [EIP, Syntax.Pop, EAX]));
  except
    Result := False;
  end;
end;

function TMiniAsmInterpreter.DoPush(AOutput: TStrings): Boolean;
begin
  Result := True;
  try
    FStack.Push(EAX);
    AOutput.Add(Format('%d: %s' + #9 + '<AC> = %d', [EIP, Syntax.Push, EAX]));
  except
    Result := False;
  end;
end;

function TMiniAsmInterpreter.DoGoSub(AOutput: TStrings; AArg: Integer): Boolean;
begin
  Result := True;
  try
    FStack.Push(EIP);
    AOutput.Add(Format('%d: %s %d', [EIP, Syntax.GoSub, AArg]));
    EIP := AArg - 1; // -1, da spter EIP wieder inkrementiert wird
  except
    Result := False;
  end;
end;

function TMiniAsmInterpreter.DoRet(AOutput: TStrings): Boolean;
begin
  Result := True;
  try
    EIP := FStack.Pop;
    AOutput.Add(Format('%d: %s', [EIP, Syntax.Ret]));
  except
    Result := False;
  end;
end;

{$ENDREGION}

{$REGION 'Getter & Setter'}

function TMiniAsmInterpreter.GetOnStackChange: TStackChangeEvent;
begin
  Result := FStack.OnStackChange;
end;

procedure TMiniAsmInterpreter.SetEAX(const Value: Integer);
begin
  Feax := Value;
  if Assigned(FOnEAXChange) then
    FOnEAXChange(Self, EAX);
end;

procedure TMiniAsmInterpreter.SetEIP(const Value: Integer);
begin
  Feip := Value;
  if Assigned(FOnEIPChange) then
    FOnEIPChange(Self, EIP);
end;

procedure TMiniAsmInterpreter.SetOnStackChange(const Value: TStackChangeEvent);
begin
  FStack.OnStackChange := Value;
end;

procedure TMiniAsmInterpreter.SetSF(const Value: Boolean);
begin
  Fsf := Value;
  if Assigned(FOnSFChange) then
    FOnSFChange(Self, SF);
end;

procedure TMiniAsmInterpreter.SetSource(ASource: TStrings);
begin
  inherited SetSource(ASource);
  EIP := 0;
  PrepareSource;
end;

procedure TMiniAsmInterpreter.SetZF(const Value: Boolean);
begin
  Fzf := Value;
  if Assigned(FOnZFChange) then
    FOnZFChange(Self, ZF);
end;

{$ENDREGION}

end.
