{******************************************************************************}
{                                                                              }
{                                ConsoleMemo                                   }
{                                                                              }
{   Component derived from TMemo imitating the behaviour of a console like     }
{   the "DOS-box"                                                              }
{                                                                              }
{   original author:   Christian Rehn aka r2c2                                 }
{                                                                              }
{   This code is licenced under the terms of the Mozilla Public Licence(MPL)   }
{   See http://www.mozilla.org/MPL/ or licence-file included in source-archive }
{   for licence terms.                                                         }
{                                                                              }
{******************************************************************************}

unit ConsoleMemo;

interface

uses
  StdCtrls, Classes, Graphics;

type
  TConsoleMemoReadMode = (rmNone, rmString, rmInteger);
  TConsoleMemo = class(TMemo)
  private
    procedure EnsureCaretPos;
  strict private
    FMinPos: Integer;
    FReadComplete: Boolean;
    FReadMode: TConsoleMemoReadMode;
  strict protected
    property MinPos: Integer read FMinPos write FMinPos;
    procedure KeyPress(var Key: Char); override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure Click; override;
    procedure Change; override;
  public
    constructor Create(AOwner: TComponent); override;
    property ReadOnly default True;
    property Color default clBlack;
    property ReadMode: TConsoleMemoReadMode read FReadMode default rmNone;
    function ReadString: string; virtual;
    function ReadInt: Integer; virtual;
  end;

implementation

uses
  Forms, SysUtils, Windows;

constructor TConsoleMemo.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  ReadOnly := True;
  FReadComplete := False;
  FReadMode := rmNone;
  Color := clBlack;
  Font.Color := clSilver;
  Font.Name := 'Courier New';
  Font.Size := 10;
  ScrollBars := ssVertical;
end;

procedure TConsoleMemo.KeyPress(var Key: Char);
begin
  if (Key = #8) and (SelStart <= FMinPos) then
    Key := #0;  // Backspace abfangen, wenn mehr gelscht wrde, als erlaubt

  inherited KeyPress(Key);
  if (FReadMode <> rmNone) and (Key = #13) then
  begin
    FReadComplete := True;
  end;
  if (FReadMode = rmInteger)
    and (
         ((not (Key in ['0'..'9', #8])) and (SelStart > MinPos))
      or ((not (Key in ['0'..'9', #8, '-'])) and (SelStart = MinPos))
    ) then
  begin
    Key := #0;       // im Integer-Mode nur Zahlen zulassen
  end;
  EnsureCaretPos;
end;

procedure TConsoleMemo.KeyDown(var Key: Word; Shift: TShiftState);
begin
  if (Key = VK_BACK) and (SelStart <= FMinPos) then
    Key := 0; // Backspace abfangen, wenn mehr gelscht wrde, als erlaubt

  inherited KeyDown(Key, Shift);
  EnsureCaretPos;
end;

procedure TConsoleMemo.Click;
begin
  inherited Click;
  EnsureCaretPos;
end;

procedure TConsoleMemo.Change;
begin
  inherited Change;
  EnsureCaretPos;   //TODO: paste --> Int
end;

function TConsoleMemo.ReadInt: Integer;
var
  s: string;
begin
  FMinPos := SelStart; // maximal editierbarer Bereich
  FReadComplete := False;
  ReadOnly := False;
  FReadMode := rmInteger;
  while (not FReadComplete) and (not Application.Terminated) do
  begin
    Application.ProcessMessages;   //TODO: WaitForSingleObject
  end;
  ReadOnly := True;
  FReadMode := rmNone;
  s := Trim(Copy(Lines.Text, FMinPos, Length(Lines.Text) - FMinPos + 1));
  Result := StrToIntDef(s, 0);
end;

procedure TConsoleMemo.EnsureCaretPos;
begin
  // Caret nur in gltigem Bereich zulassen:
  if (FReadMode <> rmNone) and (SelStart < FMinPos) then
  begin
    SelStart := FMinPos;
  end;
end;

function TConsoleMemo.ReadString: string;
var
  s: string;
begin
  FMinPos := SelStart;   // maximal editierbarer Bereich
  FReadComplete := False;
  ReadOnly := False;
  FReadMode := rmString;
  while (not FReadComplete) and (not Application.Terminated) do
  begin
    Application.ProcessMessages;  //TODO: WaitForSingleObject
  end;
  ReadOnly := True;
  FReadMode := rmNone;
  s := Trim(Copy(Lines.Text, FMinPos, Length(Lines.Text) - FMinPos + 1));
  Result := s;
end;

end.
