{******************************************************************************}
{                                                                              }
{                                MiniASMSourceWorker                           }
{                                                                              }
{   BaseClass for TMiniAsmChecker and TMiniASMInterpreter.                     }
{                                                                              }
{   original author:   Christian Rehn aka r2c2                                 }
{                                                                              }
{   This code is licenced under the terms of the Mozilla Public Licence(MPL)   }
{   See http://www.mozilla.org/MPL/ or licence-file included in surce-archive  }
{   for licence terms.                                                         }
{                                                                              }
{******************************************************************************}

unit MiniAsmSourceWorker;

interface

uses Classes, Syntax;

type
  TMiniAsmSourceWorker = class
  strict private
    FSyntax:TSyntax;
    FSource: TStrings;
  strict protected
    procedure RemoveComments; virtual;
    procedure PrepareSource; virtual;
    procedure SetSource(ASource: TStrings); virtual;
    procedure ParseLine(ALine: Integer; var AOperation: string; var AArg: string); virtual;
  public
    constructor Create(ASource: TStrings; ASyntax: TSyntax);
    destructor Destroy; override;
    property Source: TStrings read FSource write FSource;
    property Syntax :  TSyntax read FSyntax write FSyntax;
    function Execute(AOutput: TStrings): Boolean; virtual; abstract;
  end;

implementation

uses
  MiniAsmInterpreter, SysUtils;
 
constructor TMiniAsmSourceWorker.Create(ASource: TStrings; ASyntax: TSyntax);
begin
  inherited Create;
  FSource := TStringList.Create;
  if ASource <> nil then
    FSource.Assign(ASource);
  FSyntax := ASyntax;
end;

destructor TMiniAsmSourceWorker.Destroy;
begin
  FSource.Free;
  inherited Destroy;
end;

procedure TMiniAsmSourceWorker.SetSource(ASource: TStrings);
begin
  FSource.Assign(ASource);
end;

procedure TMiniAsmSourceWorker.RemoveComments;
var
  i: Integer;
  Position: Integer;
  tmp: string;
begin
  for i := 0 to FSource.Count - 1 do
  begin
    Position := Pos(';', FSource[i]);
    tmp := FSource[i];
    Delete(tmp, Position, Length(FSource[i]) - Position + 1);
    FSource[i] := tmp;
  end;
end;

procedure TMiniAsmSourceWorker.ParseLine(ALine: Integer; var AOperation: string;
  var AArg: string);
var
  PosSpace: Integer;
begin
  Source[ALine] := Trim(Source[ALine]);
  AOperation := Source[ALine]; // alles
  PosSpace := Pos(' ', AOperation); // Trennung Operation Arg
  if PosSpace > 0 then
  begin
    // Arg is alles nach dem Space:
    AArg := Copy(AOperation, PosSpace + 1, Length(AOperation) - PosSpace);
    Delete(AOperation, PosSpace, Length(AOperation) - PosSpace + 1);
  end
  else
  begin
    AArg := '';
  end;
  AOperation := Trim(AOperation);
  AArg := Trim(AArg);
end;

procedure TMiniAsmSourceWorker.PrepareSource;
var
  i: Integer;
  Operation: string;
  Arg: string;
  Labels: TStringList;
begin
  RemoveComments;

  // Leerzeilen am Ende entfernen:
  if Source.Count > 1 then
  begin
    i := Source.Count -1;
    while Trim(Source[i]) = '' do
    begin
      Source.Delete(i);
      Dec(i);
    end;
  end;

  Labels := TStringList.Create;
  try
    // Liste aller Labels samt Zeilennummer erstellen:
    for i := 0 to Source.Count - 1 do
    begin
      Source[i] := Trim(Source[i]); // Leerzeichen (am Ende) entfernen
      if Syntax.IsLabel(Source[i]) then
      begin
        Labels.AddObject(Source[i], Pointer(i));
      end;
    end;
    // Labels durch Zeilennummern ersetzen:
    for i := 0 to Source.Count - 1 do
    begin
      ParseLine(i, Operation, Arg);

      if Syntax.IsLabel(Arg) then
      begin
        Source[i] := Operation + ' '
          + IntToStr(Integer(Labels.Objects[Labels.IndexOf(Arg)]) + 1);
      end;
    end;
    // Labels entfernen:
    for i := 0 to Labels.Count - 1 do
    begin
      Source[Integer(Labels.Objects[i])] := '';
    end;
  finally
    Labels.Free;
  end;
end;

end.
