{******************************************************************************}
{                                                                              }
{                                Syntax                                        }
{                                                                              }
{   Class managing mnemonics for the MiniASM-Syntax.                           }
{                                                                              }
{   original author:   Christian Rehn aka r2c2                                 }
{                                                                              }
{   This code is licenced under the terms of the Mozilla Public Licence(MPL)   }
{   See http://www.mozilla.org/MPL/ or licence-file included in surce-archive  }
{   for licence terms.                                                         }
{                                                                              }
{******************************************************************************}

unit Syntax;

interface

type
  TOperation = (opRead, opWrite, opLoad, opStore, opAdd, opSub, opMul, opDiv,
    opJmp, opJz, opJns, opPrint, opData, opPush, opPop, opGoSub, opRet, opEnd);

  TCommand = class
  strict private
    FOperation:TOperation;
    FArg_i:Integer;
    FArg_s: string;
  public
    constructor Create(AOperation: TOperation; AArg: Integer = 0); overload;
    constructor Create(AOperation: TOperation; AArg: string); overload;
    property Operation :  TOperation read FOperation write FOperation;
    property Arg_s :  string read FArg_s write FArg_s;
    property Arg_i :  Integer read FArg_i write FArg_i;
  end;

  TSyntax = class
  strict private
    // general settings:
    FStackSize: Integer;
    FCaseSensitive: Boolean;
    // basic operatorset:
    FReadInt: string;
    FWriteInt: string;
    FLoad: string;
    FStore: string;
    FAdd: string;
    FSubstract: string;
    FMultiply: string;
    FDevide: string;
    FJmp: string;
    FJz: string;
    FJns: string;
    FLabel: string;
    FEnd: string;
    // extended operatorset:
    FPrint: string;
    FData: string;
    FPush: string;
    FPop: string;
    FGoSub: string;
    FRet: string;
  public
    // general settings:
    property StackSize: Integer read FStackSize write FstackSize;
    property CaseSensitive: Boolean read FCaseSensitive write FCaseSensitive;
    // bsic operatorset:
    property ReadInt: string read FReadInt write FReadInt;
    property WriteInt: string read FWriteInt write FWriteInt;
    property Load: string read FLoad write FLoad;
    property Store: string read FStore write FStore;
    property Add: string read FAdd write FAdd;
    property Substract: string read FSubstract write FSubstract;
    property Mutiply: string read FMultiply write FMultiply;
    property Devide: string read FDevide write FDevide;
    property Jmp: string read FJmp write FJmp;
    property Jz: string read FJz write FJz;
    property Jns: string read FJns write FJns;
    property &Label: string read FLabel write FLabel;
    property &End: string read FEnd write FEnd;
    // extended operatorset:
    property Print: string read FPrint write FPrint;
    property Data: string read FData write FData;
    property Push: string read FPush write FPush;
    property Pop: string read FPop write FPop;
    property GoSub: string read FGoSub write FGoSub;
    property Ret: string read FRet write FRet;

    procedure LoadFromFile(const AFileName: string); virtual;
    procedure SaveToFile(const AFileName: string); virtual;
    function IsCommand(const AOperation: string; const AArg: string): Boolean; virtual;
    function IsLabel(const s: string): Boolean; virtual;
    function CreateCommand(AOperation: string; AArg: string): TCommand; virtual;
  end;

implementation

uses
  IniFiles, SysUtils;

{$REGION 'TCommand'}

constructor TCommand.Create(AOperation: TOperation; AArg: Integer = 0);
begin
  inherited Create;
  FOperation := AOperation;
  FArg_i := AArg;
end;

constructor TCommand.Create(AOperation: TOperation; AArg: string);
begin
  inherited Create;
  FOperation := AOperation;
  FArg_s := AArg;
end;

{$ENDREGION}

{$REGION 'TSyntax'}

procedure TSyntax.LoadFromFile(const AFileName: string);
var
  ini: TMemIniFile;
begin
  ini := TMemIniFile.Create(AFileName);
  try
    // general settings:
    FStackSize := ini.ReadInteger('GeneralSettings', 'StackSize', 32);
    FCaseSensitive := ini.ReadBool('GeneralSettings', 'CaseSensitive', False);
    // basic operators:
    FReadInt := ini.ReadString('BasicOperators', 'ReadInt', 'read');
    FWriteInt := ini.ReadString('BasicOperators', 'WriteInt', 'writ');
    FLoad := ini.ReadString('BasicOperators', 'Load', 'lade');
    FStore := ini.ReadString('BasicOperators', 'Store', 'stor');
    FAdd := ini.ReadString('BasicOperators', 'Add', 'add+');
    FSubstract := ini.ReadString('BasicOperators', 'Substract', 'sub-');
    FMultiply := ini.ReadString('BasicOperators', 'Multiply', 'mul*');
    FDevide := ini.ReadString('BasicOperators', 'Devide', 'div/');
    FJmp := ini.ReadString('BasicOperators', 'Jmp', 'goto');
    FJz := ini.ReadString('BasicOperators', 'Jz', 'go.0');
    FJns := ini.ReadString('BasicOperators', 'Jns', 'go.p');
    FLabel := ini.ReadString('BasicOperators', 'Label', ':');
    FEnd := ini.ReadString('BasicOperators', 'End', 'end.');
    // extended operators:
    FPrint := ini.ReadString('ExtendedOperators', 'Print', 'print');
    FData := ini.ReadString('ExtendedOperators', 'Data', 'data');
    FPush := ini.ReadString('ExtendedOperators', 'Push', 'push');
    FPop := ini.ReadString('ExtendedOperators', 'Pop', 'pop');
    FGoSub := ini.ReadString('ExtendedOperators', 'GoSub', 'gosub');
    FRet := ini.ReadString('ExtendedOperators', 'Ret', 'ret');
  finally
    ini.Free;
  end;
end;

procedure TSyntax.SaveToFile(const AFileName: string);
var
  ini: TMemIniFile;
begin
  ini := TMemIniFile.Create(AFileName);
  try
    // general settings:
    ini.WriteInteger('GeneralSettings', 'StackSize', FStackSize);
    ini.WriteBool('GeneralSettings', 'CaseSensitive', FCaseSensitive);
    // basic operators:
    ini.WriteString('BasicOperators', 'ReadInt', FReadInt);
    ini.WriteString('BasicOperators', 'WriteInt', FWriteInt);
    ini.WriteString('BasicOperators', 'Load', FLoad);
    ini.WriteString('BasicOperators', 'Store', FStore);
    ini.WriteString('BasicOperators', 'Add', FAdd);
    ini.WriteString('BasicOperators', 'Substract', FSubstract);
    ini.WriteString('BasicOperators', 'Multiply', FMultiply);
    ini.WriteString('BasicOperators', 'Devide', FDevide);
    ini.WriteString('BasicOperators', 'Jmp', FJmp);
    ini.WriteString('BasicOperators', 'Jz', FJz);
    ini.WriteString('BasicOperators', 'Jns', FJns);
    ini.WriteString('BasicOperators', 'Label', FLabel);
    ini.WriteString('BasicOperators', 'End', FEnd);
    // extended operators:
    ini.WriteString('ExtendedOperators', 'Print', FPrint);
    ini.WriteString('ExtendedOperators', 'Data', FData);
    ini.WriteString('ExtendedOperators', 'Push', FPush);
    ini.WriteString('ExtendedOperators', 'Pop', FPop);
    ini.WriteString('ExtendedOperators', 'GoSub', FGoSub);
    ini.WriteString('ExtendedOperators', 'Ret', FRet);
    ini.UpdateFile;
  finally
    ini.Free;
  end;
end;

function TSyntax.IsLabel(const s: string): Boolean;
begin
  Result := ((FLabel <> '') and (Pos(FLabel, Trim(s)) = 1));
end;

function TSyntax.IsCommand(const AOperation: string; const AArg: string): Boolean;
var
  cmd: TCommand;
begin
  cmd := CreateCommand(AOperation, AArg);
  try
    Result := Assigned(cmd);
  finally
    cmd.Free;
  end;
end;

function TSyntax.CreateCommand(AOperation: string; AArg: string): TCommand;
begin
  Result := nil;
  AOperation := Trim(AOperation);
  AArg := Trim(AArg);

  if AOperation <> '' then
  begin
    if not FCaseSensitive then
    begin
      AOperation := AnsiLowerCase(AOperation);
      FReadInt := AnsiLowerCase(FReadInt);
      FWriteInt := AnsiLowerCase(FWriteInt);
      FLoad := AnsiLowerCase(FLoad);
      FStore := AnsiLowerCase(FStore);
      FAdd := AnsiLowerCase(FAdd);
      FSubstract := AnsiLowerCase(FSubstract);
      FMultiply := AnsiLowerCase(FMultiply);
      FDevide := AnsiLowerCase(FDevide);
      FJmp := AnsiLowerCase(FJmp);
      FJz := AnsiLowerCase(FJz);
      FJns := AnsiLowerCase(FJns);
      FPrint := AnsiLowerCase(FPrint);
      FData := AnsiLowerCase(FData);
      FPush := AnsiLowerCase(FPush);
      FPop := AnsiLowerCase(FPop);
      FGoSub := AnsiLowerCase(FGoSub);
      FRet := AnsiLowerCase(FRet);
    end;

    if (AOperation = FReadInt) and (StrToIntDef(AArg, -1) >= 0) then
    begin
      Result := TCommand.Create(opRead, StrToInt(AArg));
    end
    else if (AOperation = FWriteInt) and (StrToIntDef(AArg, -1) >= 0) then
    begin
      Result := TCommand.Create(opWrite, StrToInt(AArg));
    end
    else if (AOperation = FLoad) and (StrToIntDef(AArg, -1) >= 0) then
    begin
      Result := TCommand.Create(opLoad, StrToInt(AArg));
    end
    else if (AOperation = FStore) and (StrToIntDef(AArg, -1) >= 0) then
    begin
      Result := TCommand.Create(opStore, StrToInt(AArg));
    end
    else if (AOperation = FAdd) and (StrToIntDef(AArg, -1) >= 0) then
    begin
      Result := TCommand.Create(opAdd, StrToInt(AArg));
    end
    else if (AOperation = FSubstract) and (StrToIntDef(AArg, -1) >= 0) then
    begin
      Result := TCommand.Create(opSub, StrToInt(AArg));
    end
    else if (AOperation = FMultiply) and (StrToIntDef(AArg, -1) >= 0) then
    begin
      Result := TCommand.Create(opMul, StrToInt(AArg));
    end
    else if (AOperation = FDevide) and (StrToIntDef(AArg, -1) >= 0) then
    begin
      Result := TCommand.Create(opDiv, StrToInt(AArg));
    end
    else if (AOperation = FJmp) and (StrToIntDef(AArg, -1) >= 0) then
    begin
      Result := TCommand.Create(opJmp, StrToInt(AArg));
    end
    else if (AOperation = FJz) and (StrToIntDef(AArg, -1) >= 0) then
    begin
      Result := TCommand.Create(opJz, StrToInt(AArg));
    end
    else if (AOperation = FJns) and (StrToIntDef(AArg, -1) >= 0) then
    begin
      Result := TCommand.Create(opJns, StrToInt(AArg));
    end
    else if (AOperation = FPrint) and (AArg <> '') then
    begin
      Result := TCommand.Create(opPrint, AArg);
    end
    else if (AOperation = FData) and (StrToIntDef(AArg, Low(Integer)) > Low(Integer)) then
    begin
      Result := TCommand.Create(opData, StrToInt(AArg));
    end
    else if (AOperation = FPush) and (AArg = '') then
    begin
      Result := TCommand.Create(opPush);
    end
    else if (AOperation = FPop) and (AArg = '') then
    begin
      Result := TCommand.Create(opPop);
    end
    else if (AOperation = FGoSub) and (StrToIntDef(AArg, -1) >= 0) then
    begin
      Result := TCommand.Create(opGoSub, StrToInt(AArg));
    end
    else if (AOperation = FRet) and (AArg = '') then
    begin
      Result := TCommand.Create(opRet);
    end
    else if (AOperation = FEnd) and ((AArg = '') or (AArg = '0')) then
    begin
      Result := TCommand.Create(opEnd);
    end
  end;
end;

{$ENDREGION}

end.
