unit TPersistentFormPos_class;

interface

uses
  Forms;

type
  TPersistentFormPos = class(TObject)
  private
    FForm: TForm;
    FFileName: string;
    function GetShellFolder(const CSIDL: Integer): string;
    function GetAppDataDir: string;
  public
    constructor Create(AForm: TForm);
    property FileName: string read FFileName write FFileName;
    function LoadFormPos(AFileName: string = ''): Boolean;
    procedure SaveFormPos(AFileName: string = '');
    procedure RemoveFiles(AFileName: string = '');
  end;

implementation

uses
  Inifiles, ActiveX, ShlObj, Windows, SysUtils;

constructor TPersistentFormPos.Create(AForm: TForm);
begin
  inherited Create;
  FForm := AForm;
  FFileName := GetAppDataDir + Application.Title +  '\FormSettings.ini';
end;

function TPersistentFormPos.LoadFormPos(AFileName: string = ''): Boolean;
var
  ini: TMemIniFile;
begin
  if AFileName = '' then
    AFileName := FFileName
  else
    FFileName := AFileName;
  Result := FileExists(AFileName);

  if Result then
  begin
    if not DirectoryExists(ExtractFilePath(AFileName)) then
      ForceDirectories(ExtractFilePath(AFileName));
                                                 
    ini := TMemIniFile.Create(AFileName);
    try
      FForm.WindowState := TWindowState(ini.ReadInteger('FormPos', 'State',
        Integer(FForm.WindowState)));
      if FForm.WindowState <> wsMaximized then
      begin
        FForm.Top := ini.ReadInteger('FormPos', 'Top', FForm.Top);
        FForm.Left := ini.ReadInteger('FormPos', 'Left', FForm.Left);
        FForm.Width := ini.ReadInteger('FormPos', 'Width', FForm.Width);
        FForm.Height := ini.ReadInteger('FormPos', 'Height', FForm.Height);
      end;
    finally
      ini.Free;
    end;
  end;
end;

procedure TPersistentFormPos.SaveFormPos(AFileName: string = '');
var
  ini: TMemIniFile;
begin
  if AFileName = '' then
    AFileName := FFileName
  else
    FFileName := AFileName;

  if not DirectoryExists(ExtractFilePath(AFileName)) then
    ForceDirectories(ExtractFilePath(AFileName));

  ini := TMemIniFile.Create(AFileName);
  try
    ini.WriteInteger('FormPos', 'State', Integer(FForm.WindowState));
    ini.WriteInteger('FormPos', 'Top', FForm.Top);
    ini.WriteInteger('FormPos', 'Left', FForm.Left);
    ini.WriteInteger('FormPos', 'Width', FForm.Width);
    ini.WriteInteger('FormPos', 'Height', FForm.Height);
    ini.UpdateFile;
  finally
    ini.Free;
  end;
end;

procedure TPersistentFormPos.RemoveFiles(AFileName: string);
begin
  if AFileName = '' then
    AFileName := FFileName
  else
    FFileName := AFileName;

  if FileExists(AFileName) then
  begin
    DeleteFile(AFileName);
    RemoveDir(ExtractFilePath(AFileName));
  end;     
end;

function TPersistentFormPos.GetShellFolder(const CSIDL: Integer): string;
var
  pidl         : PItemIdList;
  FolderPath   : string;
  SystemFolder : Integer;
  Malloc       : IMalloc;
begin
  Malloc := nil;
  FolderPath := '';
  SHGetMalloc(Malloc);
  if Malloc = nil then
  begin
    Result := FolderPath;
    Exit;
  end;
  try
    SystemFolder := CSIDL;
    if SUCCEEDED(SHGetSpecialFolderLocation(0, SystemFolder, pidl)) then
    begin
      SetLength(FolderPath, max_path);
      if SHGetPathFromIDList(pidl, PChar(FolderPath)) then
      begin
        SetLength(FolderPath, length(PChar(FolderPath)));
      end;
    end;
    Result := FolderPath;
  finally
    Malloc.Free(pidl);
  end;
end;

function TPersistentFormPos.GetAppDataDir: string;
begin
  Result := IncludeTrailingPathDelimiter(GetShellFolder(CSIDL_APPDATA));
end;

end.
