/**
 * This file is part of InterpreterDemo. Use it for every purpose, you like.
 * Cosider this code as public domain.
 *
 * \author Christian Rehn
 */

#include "Tokenizer.h"

#include <assert.h>
#include <cstdlib> //TODO: needed?
#include <cctype>
#include <sstream>

namespace InterpreterDemo
{

	Tokenizer::Tokenizer(const string& str)
		: _str(str),
			_pos(0),
			_currentToken("")
	{

	}

	string Tokenizer::getStr() const
	{
		return _str;
	}

	void Tokenizer::setStr(const string& value)
	{
		_str = value;
		_pos = 0;
	}

	bool Tokenizer::hasNext() const
	{
		return _pos < _str.length();
	}

	Token Tokenizer::getCurrentToken() const
	{
		return _currentToken;
	}

	int Tokenizer::getCurrentNumberTokenValue() const
	{
		assert(_currentToken == "Number");

		return _currentNumberTokenValue;
	}

	string Tokenizer::getCurrentTokenAsStr() const
	{
		string result = _currentToken;
		if (_currentToken == "Number")
		{
			std::ostringstream stringstream;
			stringstream << _currentNumberTokenValue;
			result += " " + stringstream.str();
		}

		return result;
	}

	Token Tokenizer::getNextToken()
	{
		if (! hasNext())
		{
			_currentToken = Token(""); // end of string; nothing to be read
			return _currentToken;
		}

		// skip whitespace
		while (isspace(_str[_pos]))
		{
			_pos++;
		}

		switch (_str[_pos])
		{
		case '0': case '1': case '2': case '3': case '4':
		case '5': case '6': case '7': case '8': case '9':
			_currentToken = readNumber();
			break;
		case '+':
			_currentToken = readPlus();
			break;
		case '-':
			_currentToken = readMinus();
			break;
		case '*':
			_currentToken = readTimes();
			break;
		case '/':
			_currentToken = readDivide();
			break;
		case '(':
			_currentToken = readLeftParenthesis();
			break;
		case ')':
			_currentToken = readRightParenthesis();
			break;
		default:
			// template method to be implemented in subclasses
			// returns empty token by default
			_currentToken = readCustomTokens();
			
			if (_currentToken == "") // token still not recognized
			{
				throw TokenizerException(string("Unrecognized character: '")
						+ _str[_pos] + "'");
			}
		}

		return _currentToken;
	}

	Token Tokenizer::readNumber()
	{
		assert(isdigit(_str[_pos]));
		
		int numVal = 0;
		for (; _pos < _str.length() && isdigit(_str[_pos]); _pos++)
		{
			numVal = numVal * 10 + _str[_pos] - '0';
		}
		_currentNumberTokenValue = numVal;

		return Token("Number");
	}

	Token Tokenizer::readPlus()
	{
		assert(_str[_pos] == '+');

		_pos++;
		return Token("Plus");
	}

	Token Tokenizer::readMinus()
	{
		assert(_str[_pos] == '-');

		_pos++;
		return Token("Minus");
	}

	Token Tokenizer::readTimes()
	{
		assert(_str[_pos] == '*');

		_pos++;
		return Token("Times");
	}

	Token Tokenizer::readDivide()
	{
		assert(_str[_pos] == '/');

		_pos++;
		return Token("Divide");
	}

	Token Tokenizer::readLeftParenthesis()
	{
		assert(_str[_pos] == '(');

		_pos++;
		return Token("LeftParenthesis");
	}

	Token Tokenizer::readRightParenthesis()
	{
		assert(_str[_pos] == ')');

		_pos++;
		return Token("RightParenthesis");
	}

	Token Tokenizer::readCustomTokens()
	{
		return Token(""); // do nothing here
	}

}