/**
 * This file is part of InterpreterDemo. Use it for every purpose, you like.
 * Cosider this code as public domain.
 *
 * \author Christian Rehn
 */

#include "Parser.h"

#include <assert.h>

#include "Interpreter.h"

namespace InterpreterDemo
{

	Parser::Parser(Tokenizer tokenizer)
		: _tokenizer(tokenizer),
			_lookahead("")
	{
	}

	Expression* Parser::parse(const string& str)
	{
		if (str != "")
		{
			_tokenizer.setStr(str);
		}
		nextToken();
		return parseExpression();
	}

	Expression* Parser::parseExpression()
	{
		// the first term could be preceded by a sign
		Expression* firstPart = parseFirstPartOfExpression();

		while (_lookahead != "") // empty string signals end
		{
			if (_lookahead == "Plus")
			{
				nextToken();
				Expression* secondPart = parseTerm();
				firstPart = new AdditionExpression(firstPart, secondPart);
			}
			else if (_lookahead == "Minus")
			{
				nextToken();
				Expression* secondPart = parseTerm();
				firstPart = new SubtractionExpression(firstPart, secondPart);
			}
			else
			{
				throw ParserException("Unexpected token: "
						+ _tokenizer.getCurrentTokenAsStr());
			}
		}

		return firstPart;
	}

	Expression* Parser::parseFirstPartOfExpression()
	{
		if (_lookahead == "Plus")
		{
			nextToken(); // omit '+'
			return parseTerm();
		}
		else if (_lookahead == "Minus")
		{
			// omit '-' and invert next term
			nextToken();
			Expression* term = parseTerm();
			return new InversionExpression(term);
		}
		else // there is no sign
		{
			return parseTerm();
		}
	}

	Expression* Parser::parseTerm()
	{
		Expression* firstPart = parseFactor();

		while (_lookahead != "") // empty string signals end
		{
			if (_lookahead == "Times")
			{
				nextToken();
				Expression* secondPart = parseFactor();
				firstPart = new MultiplicationExpression(firstPart, secondPart);
			}
			else if (_lookahead == "Divide")
			{
				nextToken();
				Expression* secondPart = parseFactor();
				firstPart = new DivisionExpression(firstPart, secondPart);
			}
			else
			{
				break; // parsing term is done; continue parsing expression
			}
		}

		return firstPart;
	}

	Expression* Parser::parseFactor()
	{
		if (_lookahead == "LeftParenthesis")
		{
			return parseParenthesis();
		}
		return parseNumber();
	}

	Expression* Parser::parseParenthesis()
	{
		assert(_lookahead == "LeftParenthesis");

		nextToken();
		Expression* firstPart = parseFirstPartOfExpression();

		while (_lookahead != "RightParenthesis")
		{
			if (_lookahead == "Plus")
			{
				nextToken();
				Expression* secondPart = parseTerm();
				firstPart = new AdditionExpression(firstPart, secondPart);
			}
			else if (_lookahead == "Minus")
			{
				nextToken();
				Expression* secondPart = parseTerm();
				firstPart = new SubtractionExpression(firstPart, secondPart);
			}
			else if (_lookahead == "")
			{
				// end of expression but no matching right parenthesis
				throw ParserException("Right parenthesis expected, but not found.");
			}
			else
			{
				throw ParserException("Unexpected token: "
						+ _tokenizer.getCurrentTokenAsStr());
			}
		}
		nextToken(); // eat RightParenthesisToken
		
		return firstPart;
	}

	NumberExpression* Parser::parseNumber()
	{
		NumberExpression* result;

		if (_lookahead != "Number")
		{
			throw ParserException("Number expected but "
					+ _tokenizer.getCurrentTokenAsStr() + " found");
		}

		result = new NumberExpression(_tokenizer.getCurrentNumberTokenValue());
		nextToken();

		return result;
	}

	void Parser::nextToken()
	{
		_lookahead = _tokenizer.getNextToken();
	}

}