/**
 * This file is part of InterpreterDemo. Use it for every purpose, you like.
 * Cosider this code as public domain.
 *
 * \author Christian Rehn
 */

#ifndef _INTERPRETER_H
#define	_INTERPRETER_H

namespace InterpreterDemo
{

	/**
	 * \brief AbstractExpression in the interpreter pattern (GoF:243).
	 */
	class Expression
	{
	public:

		/**
		 * \brief Interprets, i.e. evaluates the whole (sub-)tree and thus the
		 * expression.
		 *
		 * This is the main part of the interpreter pattern. The interpreter
		 * pattern specifies a context parameter, which can be passed to the method.
		 * This is not neccessary here, but if the program gets enhanced, you
		 * could specify some parameters here.
		 */
		virtual int interpret() const = 0;
	};

	/**
	 * \brief TerminalExpression in the interpreter pattern (GoF:243).
	 *
	 * Represents some integer number.
	 */
	class NumberExpression : public Expression
	{
	public:
		NumberExpression(int number);

		int interpret() const;
	private:
		int _number;
	};

	/**
	 * \brief NonterminalExpression in the interpreter pattern (GoF:243).
	 *
	 * Represents an inversion of an expression, e.g. -(3*7+5).
	 */
	class InversionExpression : public Expression
	{
	public:
		InversionExpression(Expression* term);
		~InversionExpression();

		int interpret() const;
	protected:
		Expression* _term;
	};

	/**
	 * \brief NonterminalExpression in the interpreter pattern (GoF:243).
	 *
	 * Represents an addition of two expressions, e.g. (3*7)+(7*8).
	 */
	class AdditionExpression : public Expression
	{
	public:
		AdditionExpression(Expression* firstAddend, Expression* secondAddend);
		~AdditionExpression();

		int interpret() const;
	protected:
		Expression* _firstAddend;
		Expression* _secondAddend;
	};

	/**
	 * \brief NonterminalExpression in the interpreter pattern (GoF:243).
	 *
	 * Represents a subtraction of two expressions, e.g. (3*7)-(7*8).
	 */
	class SubtractionExpression : public Expression
	{
	public:
		SubtractionExpression(Expression* minuend, Expression* subtrahend);
		~SubtractionExpression();

		int interpret() const;
	protected:
		Expression* _minuend;
		Expression* _subtrahend;
	};

	/**
	 * \brief NonterminalExpression in the interpreter pattern (GoF:243).
	 *
	 * Represents a multiplication of two expressions, e.g. 3*7.
	 */
	class MultiplicationExpression : public Expression
	{
	public:
		MultiplicationExpression(Expression* firstFactor, Expression* secondFactor);
		~MultiplicationExpression();

		int interpret() const;
	protected:
		Expression* _firstFactor;
		Expression* _secondFactor;
	};

	/**
	 * \brief NonterminalExpression in the interpreter pattern (GoF:243).
	 *
	 * Represents an integer division of two expressions, e.g. 3/7 (which
	 * evaluates to 0).
	 */
	class DivisionExpression : public Expression
	{
	public:
		DivisionExpression(Expression* dividend, Expression* divisor);
		~DivisionExpression();

		int interpret() const;
	protected:
		Expression* _dividend;
		Expression* _divisor;
	};

}

#endif	/* _INTERPRETER_H */
