/**
 * This file is part of InterpreterDemo. Use it for every purpose, you like.
 * Cosider this code as public domain.
 *
 * \author Christian Rehn
 */

/**
 * \mainpage InterpreterDemo
 *
 * \section Overview
 *
 * This program shows how to implement a simple parser and interpreter using
 * an LL(1) parser and the interpreter pattern (GoF:243). The program consists
 * of three components, each of which shows something different: tokenizer,
 * shows a hook method in order to allow subclassing; parser, shows how
 * to implement an easy predictive recursive descent parser and interpreter,
 * shows the interpreter pattern.
 *
 * \section Tokenizer
 *
 * The tokenizer reads a string and scans it for tokens. The implementation is
 * relatively straightforward. In order to make Tokenizer subclassable there is
 * a hook method readCustomTokens(), which does nothing by default. It is
 * called by getNextToken(), which is a template method (GoF:325). If you want
 * to introduce a new token type, you can override this method and just return
 * the new token.
 *
 * \section Parser
 *
 * The parser uses the tokenizer to get the tokens out of a given string and
 * constructs a syntax tree consisting of Expression objects. This is
 * a simple example for a predictive recursive descent parser for an LL(1)
 * language. For every production in the EBNF representation of the language
 * there is a method, which parses it.
 *
 * Parser is not meant to be subclassed, which means there is no good way of
 * doing that. It just shows how to implement such a parser.
 *
 * \section Interpreter
 *
 * The interpreter is the esiest part of the program. It uses the interpreter
 * pattern (GoF:243) to calculate the result. For simplicity there is even no
 * "context" as described in the GoF-book but this could be easily added.
 * Just add an argument to the interpret() method.
 *
 * In order to enhance the interpreter, you can just subclass Expression and
 * implement interpret().
 *
 * \section Unit Tests
 *
 * There are some unit tests using the boost unit testing framework.
 * It's a bit quick'n'dirty, so don't expect too much. Just a few test using
 * the automatic test feature. No test suites, no fixtures. The framework is
 * directly linked in instead if referencing the shared library, etc. So pretty
 * much quick'n'dirty.
 */

#include <stdlib.h>
#include <iostream>
#include <assert.h>
#include <stdexcept>

#include "Tokenizer.h"
#include "Parser.h"
#include "Interpreter.h"

using namespace std;
using InterpreterDemo::Tokenizer;
using InterpreterDemo::Parser;
using InterpreterDemo::Expression;

int main(int argc, char** argv)
{
	if ((argc == 1) || (argc > 2))
	{
		cout << "Usage: interpreter \"<expression>\"" << endl
				<< endl
				<< "<expression> is a mathematical expression in infix-notation."
					<< endl
				<< "It may contain +, -, *, / and parentheses." << endl
				<< "Only integers are allowed." << endl
				<< endl
				<< "Example: interpreter \"42-(3*(2+7)-56)\"" << endl;
	}
	else if (argc == 2)
	{
		try
		{
			Tokenizer tokenizer(argv[1]);
			Parser parser(tokenizer);
			Expression* syntaxTree = parser.parse();
			cout << syntaxTree->interpret() << endl;
		}
		catch(runtime_error& e)
		{
			cout << e.what() << endl;
			return EXIT_FAILURE;
		}
	}
	else
	{
		// Just in case the program gets enhanced
		// and the programmer does'nt take care:
		assert(false);
	}

	return EXIT_SUCCESS;
}
